#ifndef __CCountedObject__
#define __CCountedObject__

//	===========================================================================

#include "ICountedObject.hpp"
using Exponent::Basics::ICountedObject;

//	===========================================================================

namespace Exponent
{
	namespace Basics
	{
		/**
		 * @class CCountedObject CCountedObject.hpp
		 * @brief Reference counted object
		 *
		 * A reference counted class. CCountedObjects should never be deleted\n
		 * directly, you should always call dereference on them. Dereference reduces\n
		 * the reference count, when this is 0 or below, the object is deleted\n
		 * In a default construction, CCountedObjects are <B>not</B> referenced\n
		 * If you need to use the object after creation, you should make sure\n
		 * that you make a call to referenced. If this call is not made\n
		 * you cannot guarantee that the object will still be existance\n
		 * \n
		 * @code
		 * // To use effectively do something along these lines
		 *
		 * // First we construct an object. This *does not* increment the reference count.
		 * // Even if you are the creator of the object you do not automatically count as wanting to reference it
		 * // This allows you to pass new objects in to arrays etc without having to define them properly first..
		 * // However it does rely on *YOU* remembering to referece the object if you are going to use it!
		 * // Objects are automatically deleted when the reference count reaches 0
		 * CCountedObject *object = new CCountedObject;
		 *
		 * // We are going to be processing with this, so we want to make sure that the object has a reference to it
		 * object->referenced();								// [Ref Count : 1]
		 *
		 * // Lets say we pass it to another object that is also going to do something with it
		 * CSomeObject *someObject = new CSomeObject(object);	// Increments reference count of object by calling referenced on it [Ref Count : 2]
		 * // Here is what the constructor would look like:
		 * // CSomeObject::CSomeObject(CCountedObject *object) { EXCHANGE_COUNTED_OBJECT(m_object, object); }
		 *
		 * // Do some process
		 * someObject->doSomeProcess();
		 *
		 * // Now delete someObject
		 * delete someObject;									// Decrements the reference count to object by calling dereference on it [Ref Count : 1]
		 * // Here is what the destructor would look like:
		 * // CSomeObject::~CSomeObject() { FORGET_COUNTED_OBJECT(m_object); }
		 *
		 * // Object is now still in existance.. we add it to another object
		 * CSomeObject *someObject2 = new CSomeObject(object);	// Again increments reference count [Ref Count : 2]
		 *
		 * // Now we dont want our object any more
		 * object->derefenced();								// Notice we dont delete the object, we just forget it! [Ref Count : 1]
		 *
		 * // The object is still 'alive' inside someObject2, so we can process
		 * someObject->doSomeProcess();
		 *
		 * // Now delete someObject. This actually deletes object
		 * delete someObject;									// Decrements the reference count to object, which makes [Ref Count : 0] and calls the destructor of object
		 *
		 * // object is now destoryed correctly
		 * @endcode
		 *
		 * All of this has several important implications:\n
		 * <OL>
		 *		<LI>You can pass annonymous pointers to objects that take CCountedObjects as pointers for example\n
		 *			@code
		 *			someObjectThatTakesCCountedObjects->addPointer(new CCountedObject);
		 *			@endcode
		 *		<LI>You must always make sure that when dealing with CCountedObjects you <B>always</B> use the references / dereferenced system, otherwise you will end up with hanging pointers and lost memory / memory leaks
		 *		<LI>Never, <B>ever</B> do something like this, as you will end up deleting an non dynamically assigned object:\n
		 *			@code
		 *			CMyCountedObject object;
		 *			// THIS IS VERY BAD!!! DONT DO THIS!!!
		 *			someObjectThatTakesCCountedObjects->addPointer(&object);
		 *			@endcode
		 * </OL>
		 *
		 * @date 07/09/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CCountedObject.hpp,v 1.5 2007/02/08 21:06:44 paul Exp $
		 */
		class CCountedObject : public ICountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Default construction
			 */
			CCountedObject();

			/**
			 * Construction with a class name
			 * @param className The name of the class
			 */
			CCountedObject(const char *className);

			/**
			 * Destruction
			 */
			virtual ~CCountedObject();

//	===========================================================================

			/**
			 * Increment the reference count
			 */
			virtual void referenced();

			/**
			 * Decrement the reference count. Object is deleted if reference count is <= 0
			 */
			virtual void dereference();

			/**
			 * Get the reference count
			 * @retval long The reference count
			 */
			virtual long getReferenceCount() const;

			/**
			 * Get a description of the object
			 * @param string On return is filled with the description
			 * @param size The size of the stirng
			 */
			virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

		protected:

//	===========================================================================

			long m_referenceCount;			/**< How many things point to this item? */

		};
	}
}

#endif	// End of CCountedObject.hpp